'use strict'

// The byte arrays in this file are used by the
// `parseToMessage` function test suite. Any byte block
// added to this file will automatically get picked up by
// that test suite. Thus, any byte block added here should be
// parseable, and serializable, by the generic LDAP object
// type associated with that byte block.

module.exports.abandonRequestBytes = [
  0x30, 0x06, // sequence, 6 bytes
  0x02, 0x01, 0x06, // message id (integer value "6")
  0x50, 0x01, 0x05 // abandon request protocol op (application primitive integer 5)
]

/**
 * Technically, this is nonsense. The spec does not define an abandon response.
 * We are making something up here just to test the ldapjs specific response
 * object.
 */
module.exports.abandonResponseBytes = [
  0x30, 0x0c, // start sequence, 12 bytes
  0x02, 0x01, 0x01, // message id (integer value 1)
  0x00, 0x07, // protocol op (0x61) bind response
  0x0a, 0x01, 0x00, // success result code (enumerated value 0)
  0x04, 0x00, // no matched DN (0-byte octet string)
  0x04, 0x00 // no diagnostic message (0-byte octet string)
]

/**
 * Represents a basic ADD response.
 * Taken from https://web.archive.org/web/20220630073105/https://nawilson.com/ldapv3-wire-protocol-reference-add/
 */
module.exports.addResponseBasicBytes = [
  0x30, 0x0c, // sequence, 12 bytes
  0x02, 0x01, 0x02, // message id 2
  0x69, 0x07, // add response op, 7 bytes
  0x0a, 0x01, 0x00, // success result code (enumerated value 0)
  0x04, 0x00, // no matched DN (0-byte octet string)
  0x04, 0x00 // no diagnostic message (0-byte octet string)
]

/**
 * Represents an ADD response with an error and a diagnostic message.
 * Taken from https://web.archive.org/web/20220630073105/https://nawilson.com/ldapv3-wire-protocol-reference-add/
 */
module.exports.addResponseNoSuchObjectBytes = [
  0x30, 0x81, 0x9d, // sequence, 157 bytes
  0x02, 0x01, 0x03, // message id 3
  0x69, 0x81, 0x97, // add response op, 151 bytes
  0x0a, 0x01, 0x20, // noSuchObject result code (enumerated value 32)

  0x04, 0x1d, // octet string, 29 bytes (matched dn)
  0x6f, 0x75, 0x3d, 0x50, 0x65, 0x6f, 0x70, 0x6c, // "ou=Peopl"
  0x65, 0x2c, 0x20, 0x64, 0x63, 0x3d, 0x65, 0x78, // "e, dc=ex"
  0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x20, 0x64, // "ample, d"
  0x63, 0x3d, 0x63, 0x6f, 0x6d, // "c=com"

  0x04, 0x73, // octet string, 115 bytes (diagnostic message)
  // "Entry uid=missing1, ou=missing2, ou=People, dc=example, dc=com cannot be
  // created because its parent does not exist."
  0x45, 0x6e, 0x74, 0x72, 0x79, 0x20, 0x75, 0x69,
  0x64, 0x3d, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6e,
  0x67, 0x31, 0x2c, 0x20, 0x6f, 0x75, 0x3d, 0x6d,
  0x69, 0x73, 0x73, 0x69, 0x6e, 0x67, 0x32, 0x2c,
  0x20, 0x6f, 0x75, 0x3d, 0x50, 0x65, 0x6f, 0x70,
  0x6c, 0x65, 0x2c, 0x20, 0x64, 0x63, 0x3d, 0x65,
  0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x20,
  0x64, 0x63, 0x3d, 0x63, 0x6f, 0x6d, 0x20, 0x63,
  0x61, 0x6e, 0x6e, 0x6f, 0x74, 0x20, 0x62, 0x65,
  0x20, 0x63, 0x72, 0x65, 0x61, 0x74, 0x65, 0x64,
  0x20, 0x62, 0x65, 0x63, 0x61, 0x75, 0x73, 0x65,
  0x20, 0x69, 0x74, 0x73, 0x20, 0x70, 0x61, 0x72,
  0x65, 0x6e, 0x74, 0x20, 0x64, 0x6f, 0x65, 0x73,
  0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x78, 0x69,
  0x73, 0x74, 0x2e
]

/**
 * Represents an ADD response with referrals.
 * Taken from https://web.archive.org/web/20220630073105/https://nawilson.com/ldapv3-wire-protocol-reference-add/
 */
module.exports.addResponseReferralsBytes = [
  0x30, 0x81, 0xcf, // sequence, 207 bytes
  0x02, 0x01, 0x03, // message id 3
  0x69, 0x81, 0xc9, // add response op, 201 bytes
  0x0a, 0x01, 0x0a, // referral result code (enumerated value 10)
  0x04, 0x00, // no matched dn (0-byte octet string)

  0x04, 0x2f, // octet string, 47 bytes (diagnostic message)
  // "This server is read-only. Try a different one."
  0x54, 0x68, 0x69, 0x73, 0x20, 0x73, 0x65, 0x72,
  0x76, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x72,
  0x65, 0x61, 0x64, 0x2d, 0x6f, 0x6e, 0x6c, 0x79,
  0x2e, 0x20, 0x20, 0x54, 0x72, 0x79, 0x20, 0x61,
  0x20, 0x64, 0x69, 0x66, 0x66, 0x65, 0x72, 0x65,
  0x6e, 0x74, 0x20, 0x6f, 0x6e, 0x65, 0x2e,

  0xa3, 0x81, 0x90, // referrals sequence, 144 bytes
  0x04, 0x46, // string, 70 bytes (first url)
  // "ldap://alternate1.example.com:389/uid=jdoe,ou=Remote,dc=example,dc=com"
  0x6c, 0x64, 0x61, 0x70, 0x3a, 0x2f, 0x2f, 0x61,
  0x6c, 0x74, 0x65, 0x72, 0x6e, 0x61, 0x74, 0x65,
  0x31, 0x2e, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c,
  0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3a, 0x33, 0x38,
  0x39, 0x2f, 0x75, 0x69, 0x64, 0x3d, 0x6a, 0x64,
  0x6f, 0x65, 0x2c, 0x6f, 0x75, 0x3d, 0x52, 0x65,
  0x6d, 0x6f, 0x74, 0x65, 0x2c, 0x64, 0x63, 0x3d,
  0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c,
  0x64, 0x63, 0x3d, 0x63, 0x6f, 0x6d,

  0x04, 0x46, // string, 70 bytes (second url)
  // "ldap://alternate2.example.com:389/uid=jdoe,ou=Remote,dc=example,dc=com"
  0x6c, 0x64, 0x61, 0x70, 0x3a, 0x2f, 0x2f, 0x61,
  0x6c, 0x74, 0x65, 0x72, 0x6e, 0x61, 0x74, 0x65,
  0x32, 0x2e, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c,
  0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3a, 0x33, 0x38,
  0x39, 0x2f, 0x75, 0x69, 0x64, 0x3d, 0x6a, 0x64,
  0x6f, 0x65, 0x2c, 0x6f, 0x75, 0x3d, 0x52, 0x65,
  0x6d, 0x6f, 0x74, 0x65, 0x2c, 0x64, 0x63, 0x3d,
  0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c,
  0x64, 0x63, 0x3d, 0x63, 0x6f, 0x6d
]

/**
 * Represents an ADD request with attributes.
 * Taken from https://web.archive.org/web/20220630073105/https://nawilson.com/ldapv3-wire-protocol-reference-add/
 */
module.exports.addRequestBytes = [
  0x30, 0x49, // start sequence, 73 bytes
  0x02, 0x01, 0x02, // message id 2
  0x68, 0x44, // add op, 68 bytes

  0x04, 0x11, // entry dn string, 17 bytes
  0x64, 0x63, 0x3d, 0x65, // "dc=example,dc=com"
  0x78, 0x61, 0x6d, 0x70,
  0x6c, 0x65, 0x2c, 0x64,
  0x63, 0x3d, 0x63, 0x6f,
  0x6d,

  0x30, 0x2f, // start attributes sequence, 47 bytes
  0x30, 0x1c, // start first attribute sequence, 28 bytes

  0x04, 0x0b, // string, 11 bytes
  0x6f, 0x62, 0x6a, 0x65, // "objectclass"
  0x63, 0x74, 0x63, 0x6c,
  0x61, 0x73, 0x73,
  0x31, 0x0d, // start value sequence, 13 bytes
  0x04, 0x03, 0x74, 0x6f, 0x70, // string: "top"
  0x04, 0x06, 0x64, 0x6f, 0x6d, 0x61, 0x69, 0x6e, // string: "domain"

  0x30, 0x0f, // start second attribute sequence, 15 bytes
  0x04, 0x02, 0x64, 0x63, // string: "dc"
  0x31, 0x09, // start second value sequence, 9 bytes
  0x04, 0x07, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65 // string: "example"
]

/**
 * Represents an LDAP BIND Request Message as defined by
 * RFC 4511 §4.2. It has a message id of "1" and attempts to bind
 * with simple authentication as user "uid=admin,ou=sys" with the password
 * "secret".
 */
module.exports.bindRequestBytes = [
  0x30, 0x25, // sequence, 37 bytes
  0x02, 0x01, 0x01, // message id (integer value "1")
  0x60, 0x20, // protocol op (0x60), 32 bytes
  0x02, 0x01, 0x03, // version (integer value "3")
  0x04, 0x13, // string, 19 bytes
  0x75, 0x69, 0x64, 0x3d, // "uid="
  0x61, 0x64, 0x6d, 0x69, 0x6e, 0x2c, // "admin,"
  0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, // "ou=sys"
  0x74, 0x65, 0x6d, // "tem"
  0x80, 0x06, // auth choice 0 ("simple"), 6 bytes
  0x73, 0x65, 0x63, 0x72, 0x65, 0x74 // "secret"
]

/**
 * Represents an anonymous BIND request.
 */
module.exports.bindRequestAnonymousBytes = [
  0x30, 0x0c,
  0x02, 0x01, 0x01,
  0x60, 0x07,
  0x02, 0x01, 0x03,
  0x04, 0x00,
  0x80, 0x00
]

/**
 * Represents a BIND response with attributes.
 * Taken from https://web.archive.org/web/20220518232654/https://nawilson.com/ldapv3-wire-protocol-reference-bind/
 */
module.exports.bindResponseBytes = [
  0x30, 0x0c, // start sequence, 12 bytes
  0x02, 0x01, 0x01, // message id (integer value 1)
  0x61, 0x07, // protocol op (0x61) bind response
  0x0a, 0x01, 0x00, // success result code (enumerated value 0)
  0x04, 0x00, // no matched DN (0-byte octet string)
  0x04, 0x00 // no diagnostic message (0-byte octet string)
]

/**
 * Represents a COMPARE request with attributes.
 * Taken from https://web.archive.org/web/20220630080454/https://nawilson.com/ldapv3-wire-protocol-reference-compare/
 */
module.exports.compareRequestBytes = [
  0x30, 0x45, // start sequence, 69 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x6e, 0x40, // protocol op (0x6e) compare request

  0x04, 0x24, // string (target entry dn), 36 bytes
  // "uid=jdoe,ou=People,dc=example,dc=com")
  0x75, 0x69, 0x64, 0x3d, 0x6a, 0x64, 0x6f, 0x65,
  0x2c, 0x6f, 0x75, 0x3d, 0x50, 0x65, 0x6f, 0x70,
  0x6c, 0x65, 0x2c, 0x64, 0x63, 0x3d, 0x65, 0x78,
  0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63,
  0x3d, 0x63, 0x6f, 0x6d,

  0x30, 0x18, // sequence, 24 bytes
  0x04, 0x0c, // string (attribute name), 12 bytes
  // "employeeType"
  0x65, 0x6d, 0x70, 0x6c,
  0x6f, 0x79, 0x65, 0x65,
  0x54, 0x79, 0x70, 0x65,

  0x04, 0x08, // string (assertion value), 8 bytes
  // "salaried"
  0x73, 0x61, 0x6c, 0x61,
  0x72, 0x69, 0x65, 0x64
]

/**
 * Represents a COMPARE response with attributes.
 * Taken from https://web.archive.org/web/20220630080454/https://nawilson.com/ldapv3-wire-protocol-reference-compare/
 */
module.exports.compareResponseBytes = [
  0x30, 0x0c, // start sequence, 12 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x6f, 0x07, // protocol op (0x6f), 7 bytes
  0x0a, 0x01, 0x06, // compareTrue result code (enumerated value 6)
  0x04, 0x00, // No matched DN (0-byte octet string)
  0x04, 0x00 // No diagnostic message (0-byte octet string)
]

/**
 * Represents a DELETE request with controls supplied for recursive removal.
 * Taken from https://web.archive.org/web/20220629203240/https://ldap.com/ldapv3-wire-protocol-reference-ldap-message/#ldapmessage-example
 */
module.exports.deleteRequestBytes = [
  0x30, 0x35, // start sequence, 53 bytes
  0x02, 0x01, 0x05, // message id 5
  0x4a, 0x11, // delete request op (entry), 17 bytes
  0x64, 0x63, 0x3d, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, // dc=exampl
  0x65, 0x2c, 0x64, 0x63, 0x3d, 0x63, 0x6f, 0x6d, // e,dc=com

  0xa0, 0x1d, // start sequence (controls), 29 bytes
  0x30, 0x1b, // start first control sequence, 27 bytes
  0x04, 0x16, 0x31, 0x2e, // control OID (octet string):
  0x32, 0x2e, 0x38, 0x34, // 1.2.840.113556.1.4.805
  0x30, 0x2e, 0x31, 0x31,
  0x33, 0x35, 0x35, 0x36,
  0x2e, 0x31, 0x2e, 0x34,
  0x2e, 0x38, 0x30, 0x35,
  0x01, 0x01, 0xff // control criticality (boolean true)
]

/**
 * Represents a DELETE response.
 * Taken from https://web.archive.org/web/20220518193408/https://nawilson.com/ldapv3-wire-protocol-reference-delete/
 */
module.exports.deleteResponseBytes = [
  0x30, 0x0c, // start sequence, 12 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x6b, 0x07, // protocol op (0x6b), 7 bytes
  0x0a, 0x01, 0x00, // success result code (enumerated value 0)
  0x04, 0x00, // no matched DN (0-byte octet string)
  0x04, 0x00 // no diagnostic message (0-byte octet string)
]

/**
 * Represents a simple EXTENSION request with a string name
 * and string value.
 */
module.exports.extensionNameAndValueRequestBytes = [
  0x30, 0x18, // start sequence, 24 bytes
  0x02, 0x01, 0x01, // message ID (integer value 1)
  0x77, 0x13, // protocol op (0x77), 19 bytes

  0x80, 0x09, // string (request name), 9 bytes
  // "1.3.6.1.1"
  0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x31,

  0x81, 0x06, // string (request value), 6 bytes
  // "foobar"
  0x66, 0x6f, 0x6f, 0x62, 0x61, 0x72
]

/**
 * Represents a simple EXTENSION request with a string name
 * and no value.
 */
module.exports.extensionNameAndNoValueRequestBytes = [
  0x30, 0x10, // start sequence, 16 bytes
  0x02, 0x01, 0x01, // message ID (integer value 1)
  0x77, 0x0b, // protocol op (0x77), 11 bytes

  0x80, 0x09, // string (request name), 9 bytes
  // "1.3.6.1.1"
  0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x31
]

/**
 * Represents an EXTENSION request to modify a password.
 * Does not include a new password.
 * Taken from https://web.archive.org/web/20220518193613/https://nawilson.com/ldapv3-wire-protocol-reference-extended/
 */
module.exports.extensionChangePasswordRequestBytes = [
  0x30, 0x53, // start sequence, 83 bytes
  0x02, 0x01, 0x01, // message ID (integer value 1)
  0x77, 0x4e, // protocol op (0x77), 78 bytes

  0x80, 0x17, // string (request name; "oid"), 23 bytes
  // "1.3.6.1.4.1.4203.1.11.1"
  0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, // The extended request OID
  0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33,
  0x2e, 0x31, 0x2e, 0x31, 0x31, 0x2e, 0x31,

  0x81, 0x33, // sequence (request value), 51 bytes
  0x30, 0x31, // sequence, 49 bytes

  0x80, 0x24, // extension specific string (entry), 36 bytes
  // "uid=jdoe,ou=People,dc=example,dc=com"
  0x75, 0x69, 0x64, 0x3d, 0x6a, 0x64, 0x6f, 0x65,
  0x2c, 0x6f, 0x75, 0x3d, 0x50, 0x65, 0x6f, 0x70,
  0x6c, 0x65, 0x2c, 0x64, 0x63, 0x3d, 0x65, 0x78,
  0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63,
  0x3d, 0x63, 0x6f, 0x6d,

  0x81, 0x09, // extension specific string (value), 9 bytes
  // "secret123"
  0x73, 0x65, 0x63, 0x72,
  0x65, 0x74, 0x31, 0x32,
  0x33
]

/**
 * Represents an EXTENSION request to modify a password.
 * Only provides the new password.
 */
module.exports.extensionChangePasswordWithNewPasswordBytes = [
  0x30, 0x2d, // start sequence, 45 bytes
  0x02, 0x01, 0x01, // message ID (integer value 1)
  0x77, 0x28, // protocol op (0x77), 40 bytes

  0x80, 0x17, // string (request name; "oid"), 23 bytes
  // "1.3.6.1.4.1.4203.1.11.1"
  0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, // The extended request OID
  0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33,
  0x2e, 0x31, 0x2e, 0x31, 0x31, 0x2e, 0x31,

  0x81, 0x0d, // sequence (request value), 13 bytes
  0x30, 0x0b, // sequence, 11 bytes

  0x82, 0x09, // extension specific string (value), 9 bytes
  // "secret123"
  0x73, 0x65, 0x63, 0x72,
  0x65, 0x74, 0x31, 0x32,
  0x33
]

/**
 * Represents an EXTENSION response that is an unsolicited response.
 * Taken from https://web.archive.org/web/20220518193613/https://nawilson.com/ldapv3-wire-protocol-reference-extended/
 */
module.exports.extensionDisconnectionNotificationResponseBytes = [
  0x30, 0x49, // start sequence, 73 bytes
  0x02, 0x01, 0x00, // message ID (integer value 0)
  0x78, 0x44, // protocol op (0x78), 68 bytes
  0x0a, 0x01, 0x34, // unavailable result code (enumerated value 52)
  0x04, 0x00, // no matched DN (0-byte octet string)

  0x04, 0x25, // string (diagnostic message), 37 bytes
  // "The Directory Server is shutting down"
  0x54, 0x68, 0x65, 0x20, 0x44, 0x69, 0x72, 0x65,
  0x63, 0x74, 0x6f, 0x72, 0x79, 0x20, 0x53, 0x65,
  0x72, 0x76, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20,
  0x73, 0x68, 0x75, 0x74, 0x74, 0x69, 0x6e, 0x67,
  0x20, 0x64, 0x6f, 0x77, 0x6e,

  0x8a, 0x16, // string (oid)
  // "1.3.6.1.4.1.1466.20036"
  0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e,
  0x34, 0x2e, 0x31, 0x2e, 0x31, 0x34, 0x36, 0x36,
  0x2e, 0x32, 0x30, 0x30, 0x33, 0x36
]

/**
 * Represents an EXTENSION request for a Who Am I? request.
 * Taken from https://www.rfc-editor.org/rfc/rfc4532#section-2.1
 */
module.exports.extensionWhoAmIRequestBytes = [
  0x30, 0x1e, // start sequence, 30 bytes
  0x02, 0x01, 0x02, // message id, 2
  0x77, 0x19, // protocol op (0x77), 25 bytes
  0x80, 0x17, // string (oid), 23 bytes
  // "1.3.6.1.4.1.4203.1.11.3"
  0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31,
  0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32,
  0x30, 0x33, 0x2e, 0x31, 0x2e, 0x31, 0x31,
  0x2e, 0x33
]

/**
 * Represents a CANCEL request as described in
 * https://www.rfc-editor.org/rfc/rfc3909#section-2.1.
 */
module.exports.extensionCancelRequestBytes = [
  0x30, 0x19, // start sequence, 25 bytes
  0x02, 0x01, 0x02, // message id, 2
  0x77, 0x14, // protocol op (0x77), 20 bytes

  0x80, 0x0b, // string (oid), 11 bytes
  0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e,
  0x31, 0x2e, 0x31, 0x2e, 0x38,

  0x81, 0x05, // value sequence, 5 bytes
  0x30, 0x03, // sequence, 3 bytes
  0x02, 0x01, 0x01 // message id 1
]

/**
 * Represents a CANCEL response as described in
 * https://www.rfc-editor.org/rfc/rfc3909#section-2.2.
 */
module.exports.extensionCancelResponseBytes = [
  0x30, 0x0c, // sequence, 12 bytes
  0x02, 0x01, 0x02, // message id (integer 2)
  0x78, 0x07, // protocol op (0x78), 7 bytes
  0x0a, 0x01, 0x76, // status code, 118 (canceled)
  0x04, 0x00, // no matched dn
  0x04, 0x00 // no diagnostic message
]

/**
 * Represents a Start TLS request as described in
 * https://www.rfc-editor.org/rfc/rfc4511.html#section-4.14.1.
 *
 * Taken from
 * https://web.archive.org/web/20220518193613/https://nawilson.com/ldapv3-wire-protocol-reference-extended/
 */
module.exports.extenstionStartTLSRequestBytes = [
  0x30, 0x1d, // sequence, 29 bytes
  0x02, 0x01, 0x01, // message id (integer value 1)
  0x77, 0x18, // protocol op (0x77), 24 bytes

  0x80, 0x16, // name sequence, 22 bytes
  // "1.3.6.1.4.1.1466.20037"
  0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e,
  0x34, 0x2e, 0x31, 0x2e, 0x31, 0x34, 0x36, 0x36,
  0x2e, 0x32, 0x30, 0x30, 0x33, 0x37
]

/**
 * Represents a Start TLS response as described in
 * https://www.rfc-editor.org/rfc/rfc4511.html#section-4.14.2.
 *
 * Taken from
 * https://web.archive.org/web/20220518193613/https://nawilson.com/ldapv3-wire-protocol-reference-extended/
 */
module.exports.extensionStartTLSResponseBytes = [
  0x30, 0x24, // sequence, 36 bytes
  0x02, 0x01, 0x01, // message id (integer value 1)
  0x78, 0x1f, // protocol op (0x78), 31 bytes
  0x0a, 0x01, 0x00, // success result code (enumerated value 0)
  0x04, 0x00, // no matched dn (0-byte octet string)
  0x04, 0x00, // do diagnostic message (0-byte octet string)

  0x8a, 0x16, // value string, 22 bytes
  // "1.3.6.1.4.1.1466.20037"
  0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e,
  0x34, 0x2e, 0x31, 0x2e, 0x31, 0x34, 0x36, 0x36,
  0x2e, 0x32, 0x30, 0x30, 0x33, 0x37
]

/**
 * Represents a MODIFY request.
 * Taken from https://web.archive.org/web/20220518184303/https://nawilson.com/ldapv3-wire-protocol-reference-modify/.
 */
module.exports.modifyRequestBytes = [
  0x30, 0x81, 0x80, // sequence start, 128 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x66, 0x7b, // protocol op (0x66), 123 bytes

  0x04, 0x24, // string, 36 bytes
  // "uid=jdoe,ou=People,dc=example,dc=com"
  0x75, 0x69, 0x64, 0x3d, 0x6a, 0x64, 0x6f, 0x65,
  0x2c, 0x6f, 0x75, 0x3d, 0x50, 0x65, 0x6f, 0x70,
  0x6c, 0x65, 0x2c, 0x64, 0x63, 0x3d, 0x65, 0x78,
  0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63,
  0x3d, 0x63, 0x6f, 0x6d,

  0x30, 0x53, // start sequence (modifications), 83 bytes

  0x30, 0x18, // sequence, 24 bytes
  0x0a, 0x01, 0x01, // delete modification type (enumerated value 1)
  0x30, 0x13, // sequence, 19 bytes
  0x04, 0x09, // string, 9 bytes
  // "givenName"
  0x67, 0x69, 0x76, 0x65,
  0x6e, 0x4e, 0x61, 0x6d,
  0x65,
  0x31, 0x06, // string, 6 bytes
  // "John"
  0x04, 0x04, 0x4a,
  0x6f, 0x68, 0x6e,

  0x30, 0x1c, // sequence, 28 bytes
  0x0a, 0x01, 0x00, // add modification type (enumerated value 0)
  0x30, 0x17, // sequence, 23 bytes
  0x04, 0x09, // string, 9 bytes
  // "givenName"
  0x67, 0x69, 0x76, 0x65,
  0x6e, 0x4e, 0x61, 0x6d,
  0x65,
  0x31, 0x0a, // string, 10 bytes
  // "Jonathan"
  0x04, 0x08, 0x4a, 0x6f, 0x6e,
  0x61, 0x74, 0x68, 0x61, 0x6e,

  0x30, 0x19, // sequence, 25 bytes
  0x0a, 0x01, 0x02, // replace modification type (enumerated value 2)
  0x30, 0x14, // sequence, 20 bytes
  0x04, 0x02, // string, 2 bytes
  // "cn"
  0x63, 0x6e,
  0x31, 0x0e, // string, 14 bytes
  // "Jonathan Doe"
  0x04, 0x0c, 0x4a, 0x6f,
  0x6e, 0x61, 0x74, 0x68,
  0x61, 0x6e, 0x20, 0x44,
  0x6f, 0x65
]

/**
 * Represents a MODIFY response.
 * Taken from https://web.archive.org/web/20220518184303/https://nawilson.com/ldapv3-wire-protocol-reference-modify/.
 */
module.exports.modifyResponseBytes = [
  0x30, 0x0c, // sequence, 12 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x67, 0x07, // protocol op (0x67), 7 bytes
  0x0a, 0x01, 0x00, // success result code (enumerated value 0)
  0x04, 0x00, // no matched dn
  0x04, 0x00 // no diagnostic message
]

/**
 * Represents a MODIFYDN request.
 * Taken from https://web.archive.org/web/20220630073359/https://nawilson.com/ldapv3-wire-protocol-reference-modify-dn/.
 */
module.exports.modifyDnRequestBytes = [
  0x30, 0x58, // sequence, 88 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x6c, 0x53, // protocol op (0x6c), 83 bytes

  0x04, 0x24, // string, 36 bytes
  // "uid=jdoe,ou=People,dc=example,dc=com"
  0x75, 0x69, 0x64, 0x3d, 0x6a, 0x64, 0x6f, 0x65,
  0x2c, 0x6f, 0x75, 0x3d, 0x50, 0x65, 0x6f, 0x70,
  0x6c, 0x65, 0x2c, 0x64, 0x63, 0x3d, 0x65, 0x78,
  0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63,
  0x3d, 0x63, 0x6f, 0x6d,

  0x04, 0x0c, // string, 12 bytes
  // "uid=john.doe"
  0x75, 0x69, 0x64, 0x3d,
  0x6a, 0x6f, 0x68, 0x6e,
  0x2e, 0x64, 0x6f, 0x65,

  0x01, 0x01, 0xff, // Delete the old RDN value (boolean true)

  0x80, 0x1a, // context specific string, 26 bytes
  // "ou=Users,dc=example,dc=com"
  0x6f, 0x75, 0x3d, 0x55, 0x73, 0x65, 0x72, 0x73,
  0x2c, 0x64, 0x63, 0x3d, 0x65, 0x78, 0x61, 0x6d,
  0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63, 0x3d, 0x63,
  0x6f, 0x6d
]

/**
 * Represents a MODIFYDN response.
 * Taken from https://web.archive.org/web/20220630073359/https://nawilson.com/ldapv3-wire-protocol-reference-modify-dn/.
 */
module.exports.modifyDnResponseBytes = [
  0x30, 0x0c, // sequence, 12 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x6d, 0x07, // protocol op (0x6d), 7 bytes
  0x0a, 0x01, 0x00, // success result code (enumerated value 0)
  0x04, 0x00, // no matched dn
  0x04, 0x00 // no diagnostic message
]

/**
 * Represents an UNBIND request.
 * Taken from https://web.archive.org/web/20220518231541/https://nawilson.com/ldapv3-wire-protocol-reference-unbind/.
 */
module.exports.unbindRequestBytes = [
  0x30, 0x05, // sequence, 5 bytes
  0x02, 0x01, 0x03, // message id (integer value 3)
  0x42, 0x00 // protocol op (0x42), 0 bytes
]

/**
 * Represents a SEARCHREQUEST request.
 * Taken from https://web.archive.org/web/20220518215838/https://nawilson.com/ldapv3-wire-protocol-reference-search/.
 */
module.exports.searchRequestBytes = [
  0x30, 0x56, // sequence, 86 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x63, 0x51, // protocol op, 81 bytes

  0x04, 0x11, // string sequence, 17 bytes
  // "dc=example,dc=com"
  0x64, 0x63, 0x3d, 0x65,
  0x78, 0x61, 0x6d, 0x70,
  0x6c, 0x65, 0x2c, 0x64,
  0x63, 0x3d, 0x63, 0x6f,
  0x6d,

  0x0a, 0x01, 0x02, // wholeSubtree scope (enumerated value 2)

  0x0a, 0x01, 0x00, // neverDerefAliases (enumerated value 0)

  0x02, 0x02, 0x03, 0xe8, // size limit (integer value 1000)

  0x02, 0x01, 0x1e, // time limit (integer value 30)

  0x01, 0x01, 0x00, // typesOnly (boolean false)

  0xa0, 0x24, // begin AND filter, 36 bytes
  0xa3, 0x15, // begin an EQUALITY filter, 21 bytes
  0x04, 0x0b, // string, 11 bytes
  // "objectClass"
  0x6f, 0x62, 0x6a, 0x65,
  0x63, 0x74, 0x43, 0x6c,
  0x61, 0x73, 0x73,
  0x04, 0x06, // string, 6 bytes
  // "person"
  0x70, 0x65, 0x72, 0x73,
  0x6f, 0x6e, // The assertion value (octet string "person")
  0xa3, 0x0b, // begin an EQUALITY filter, 11 bytes
  0x04, 0x03, // string, 3 bytes
  // "uid"
  0x75, 0x69, 0x64,
  0x04, 0x04, // string, 4 bytes
  // "jdoe"
  0x6a, 0x64, 0x6f, 0x65,

  0x30, 0x06, // begin the set of requested attributes, 6 bytes
  0x04, 0x01, // string, 1 byte
  // "*"
  0x2a,
  0x04, 0x01, // string, 1 byte
  // "+"
  0x2b
]

/**
 * Represents a SEARCHRESULTENTRY response.
 * Taken from https://web.archive.org/web/20220518215838/https://nawilson.com/ldapv3-wire-protocol-reference-search/.
 */
module.exports.searchResultEntryBytes = [
  0x30, 0x49, // sequence, 73 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x64, 0x44, // protocol op (0x64), 68 bytes

  0x04, 0x11, // string, 17 bytes
  // "dc=example,dc=com"
  0x64, 0x63, 0x3d, 0x65,
  0x78, 0x61, 0x6d, 0x70,
  0x6c, 0x65, 0x2c, 0x64,
  0x63, 0x3d, 0x63, 0x6f,
  0x6d,

  0x30, 0x2f, // sequence, 47 bytes (attributes list)
  0x30, 0x1c, // sequence, 28 bytes (first attribute)
  0x04, 0x0b, // string, 11 bytes
  // "objectClass"
  0x6f, 0x62, 0x6a, 0x65,
  0x63, 0x74, 0x43, 0x6c,
  0x61, 0x73, 0x73,
  0x31, 0x0d, // set sequence, 13 bytes
  0x04, 0x03, // string, 3 bytes
  // "top"
  0x74, 0x6f, 0x70,
  0x04, 0x06, // string, 6 bytes
  // "domain"
  0x64, 0x6f, 0x6d, 0x61, 0x69, 0x6e,

  0x30, 0x0f, // sequence, 15 bytes (second attribute)
  0x04, 0x02, // string, 2 bytes
  // "dc"
  0x64, 0x63,
  0x31, 0x09, // set sequence, 9 bytes
  0x04, 0x07, // string,  7 bytes
  // "example"
  0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65
]

/**
 * Represents a SEARCHRESULTENTRY response that only returns attribute names.
 * Taken from https://web.archive.org/web/20220518215838/https://nawilson.com/ldapv3-wire-protocol-reference-search/.
 */
module.exports.searchResultEntryNoValuesBytes = [
  0x30, 0x33, // sequence, 51 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x64, 0x2e, // protocol op (0x64), 46 bytes

  0x04, 0x11, // string, 17 bytes
  // "dc=example,dc=com"
  0x64, 0x63, 0x3d, 0x65,
  0x78, 0x61, 0x6d, 0x70,
  0x6c, 0x65, 0x2c, 0x64,
  0x63, 0x3d, 0x63, 0x6f,
  0x6d,

  0x30, 0x19, // sequence, 25 bytes (attributes list)
  0x30, 0x0f, // sequence, 15 bytes (first attribute)
  0x04, 0x0b, // string, 11 bytes
  // "objectClass"
  0x6f, 0x62, 0x6a, 0x65,
  0x63, 0x74, 0x43, 0x6c,
  0x61, 0x73, 0x73,
  0x31, 0x00, // sequence, 0 bytes

  0x30, 0x06, // sequence, 6 bytes (second attribute)
  0x04, 0x02, // string, 2 bytes
  // "dc"
  0x64, 0x63,
  0x31, 0x00 // sequence, 0 bytes
]

/**
 * Represents a SEARCHRESULTREFERENCE response.
 * Taken from https://web.archive.org/web/20220518215838/https://nawilson.com/ldapv3-wire-protocol-reference-search/.
 */
module.exports.searchResultReferenceBytes = [
  0x30, 0x6d, // sequence, 109 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x73, 0x68, // protocol op (0x73), 104 bytes

  0x04, 0x32, // string, 50 bytes
  // "ldap://ds1.example.com:389/dc=example,dc=com??sub?"
  0x6c, 0x64, 0x61, 0x70, 0x3a, 0x2f, 0x2f, 0x64,
  0x73, 0x31, 0x2e, 0x65, 0x78, 0x61, 0x6d, 0x70,
  0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3a, 0x33,
  0x38, 0x39, 0x2f, 0x64, 0x63, 0x3d, 0x65, 0x78,
  0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63,
  0x3d, 0x63, 0x6f, 0x6d, 0x3f, 0x3f, 0x73, 0x75,
  0x62, 0x3f,

  0x04, 0x32, // string, 50 bytes
  // "ldap://ds2.example.com:389/dc=example,dc=com??sub?"
  0x6c, 0x64, 0x61, 0x70, 0x3a, 0x2f, 0x2f, 0x64,
  0x73, 0x32, 0x2e, 0x65, 0x78, 0x61, 0x6d, 0x70,
  0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3a, 0x33,
  0x38, 0x39, 0x2f, 0x64, 0x63, 0x3d, 0x65, 0x78,
  0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63,
  0x3d, 0x63, 0x6f, 0x6d, 0x3f, 0x3f, 0x73, 0x75,
  0x62, 0x3f
]

/**
 * Represents a SEARCHRESULTDONE response.
 * Taken from https://web.archive.org/web/20220518215838/https://nawilson.com/ldapv3-wire-protocol-reference-search/.
 */
module.exports.searchResultDoneBytes = [
  0x30, 0x0c, // sequence, 12 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x65, 0x07, // protocol op (0x65), 7 bytes
  0x0a, 0x01, 0x00, // success result code (enumerated value 0)
  0x04, 0x00, // no matched DN (0-byte octet string)
  0x04, 0x00 // no diagnostic message (0-byte octet string)
]

/**
 * Represents an INTERMEDIATE response with a name and value.
 */
module.exports.intermediateResponseBytes = [
  0x30, 0x11, // sequence, 17 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x79, 0x0c, // protocol op (0x79), 12 bytes

  0x80, 0x5, // string, 5 bytes
  // "1.2.3"
  0x31, 0x2e, 0x32, 0x2e, 0x33,

  0x81, 0x03, // string, 3 bytes
  // "foo"
  0x66, 0x6f, 0x6f
]

/**
 * Represents an INTERMEDIATE response with a name but no value.
 */
module.exports.intermediateResponseNoValueBytes = [
  0x30, 0x0c, // sequence, 12 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x79, 0x07, // protocol op (0x79), 12 bytes

  0x80, 0x5, // string, 5 bytes
  // "1.2.3"
  0x31, 0x2e, 0x32, 0x2e, 0x33
]

/**
 * Represents an INTERMEDIATE response with a value but no name.
 */
module.exports.intermediateResponseNoNameBytes = [
  0x30, 0x0a, // sequence, 10 bytes
  0x02, 0x01, 0x02, // message id (integer value 2)
  0x79, 0x05, // protocol op (0x79), 5 bytes

  0x81, 0x03, // string, 3 bytes
  // "foo"
  0x66, 0x6f, 0x6f
]
